/***************************************************************************/
/*  BURP                                                                   */
/*     - Bi-directional Useless Reference Ping                             */
/*                                                                         */
/*  Jill T. Bodine                                                         */
/*  John C. Broughton III                                                  */
/*  Erik J. Erikson                                                        */
/*                                                                         */
/*  A sample application that demonstrates CPI-C full-duplex conversations */
/*  and non-blocking calls.                                                */
/***************************************************************************/

#include "cpic.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "getopt.h"

#define  MAX_BUFFER_SIZE      (32763)
#define  DEFAULT_DATA_SIZE    (100)
#define  MODE_NAME            "#INTER"
#define  PARTNER_TP_NAME      "BURPD"
#define  MAX_TP_NAME_LEN      (64+1)
#define  MAX_PLU_NAME_LEN     (17+1)
#define  MAX_MODE_NAME_LEN    (8+1)
#define  BLANK_SYM_DEST_NAME  "        "
#define  BURP_MESSAGE         "Aah!  Thanks, I needed that!"

void process_arguments( int argc, char *argv[] );

CM_INT32 plu_name_length = 0;
unsigned char partner_lu_name[MAX_PLU_NAME_LEN];

char cmd_line[] = "\nBURP partner_lu_name";
char explain[] =
"The partner LU name must be either an LU alias or a fully-qualified LU name.";
unsigned short data_size = DEFAULT_DATA_SIZE;
int number_of_iterations = 1;

void main( int argc, char *argv[])
{
    /* Variables used for CPI-C calls */
    unsigned char conv_id[8];               /* CPI-C conversation ID        */
    CM_INT32 rc;                            /* CPI-C return code            */
    CM_INT32 length;                        /* General length variable      */
    CM_INT32 rts_received;                  /* Request to send rcvd         */
    CM_INT32 max_rcv_len = MAX_BUFFER_SIZE; /* Max rcv length on CMRCV      */
    CM_INT32 data_received;                 /* Data rcv parm from CMRCV     */
    CM_INT32 received_len;                  /* Amount of data rcvd on CMRCV */
    CM_INT32 status_received;               /* Status from CMRCV            */
    unsigned char * buffer = NULL;          /* Data buffer for send         */
    unsigned char * receive_buffer = NULL;  /* Data buffer for receive      */

    /* Variables for the CPI-C Set calls */
    unsigned char ptp_name[ MAX_TP_NAME_LEN ];    /* Partner's TP name       */
    unsigned char mode_name[ MAX_MODE_NAME_LEN ]; /* Mode name for conv      */
    CM_CONVERSATION_SECURITY_TYPE security_type;  /* Security type for conv  */
    CM_CONVERSATION_TYPE conv_type;               /* Conversation type       */
    CM_SEND_TYPE send_type;                       /* How to handle send data */
    CM_DEALLOCATE_TYPE deallocate_type;           /* How to end the conv     */
    CM_SEND_RECEIVE_MODE send_receive_mode;       /* FDX or HDX              */

    /* Variables used for non-blocking */
    CM_OUTSTANDING_OP_ID  send_queue_op_id;        /* used for non-blocking  */
                                                   /*  on the send queue     */
    CM_OUTSTANDING_OP_ID  receive_queue_op_id;     /* used for non-blocking  */
                                                   /*  on the receive queue  */
    CM_RETURN_CODE        send_rc;                 /* return code for calls  */
                                                   /*  on the send queue     */
    CM_RETURN_CODE        receive_rc;              /* return code for calls  */
                                                   /*  on the receive queue  */
    CM_CONVERSATION_QUEUE conv_queue;              /* which conversation Q?  */
    CM_PROCESSING_MODE    proc_mode;               /* blocking vs nonblcking */
    XC_USER_FIELD         user_field;              /* user field for         */
                                                   /*  nonblocking calls     */
    CM_INT32              outstanding_op_id_count; /* how many calls to wait */
                                                   /*  for completion on     */
    CM_INT32              timeout;                 /* timeout for cmwcmp     */
    CM_INT32              outstanding_op_id_list[2]; /* op id's for          */
                                                     /*  outstanding calls   */
    CM_INT32              output_op_id_count;      /* how many calls         */
                                                   /*  completed             */
    CM_INT32              output_op_id_list[2];    /* op id's for completed  */
                                                   /*  calls                 */
    XC_USER_FIELD         output_user_field_list[2]; /* user fields for      */
                                                     /*  completed calls     */

    
    /*
     * Say hello.
     */
    puts( "BURP - Bi-directional Useless Reference Ping." );
    puts( "  Jill T. Bodine" );
    puts( "  John C. Broughton III" );
    puts( "  Erik J. Erikson\n\n" );
    puts( "pat pat pat...\n" );

    /*
     * Find out what partner LU we are supposed to ping. This routine
     * uses getopt(), recommended for all parameter parsing!
     */
    process_arguments( argc, argv );

    /*
     * Allocate send and receive data buffer and initialize the
     * data area with zeroes. This is NOT the best way to do it
     * in OS/2. We should be allocating shared memory to avoid a
     * data copy, as CPI-C must pass our data to APPC via shared memory.
     */
    receive_buffer = malloc( (unsigned short)MAX_BUFFER_SIZE );
    if (NULL == receive_buffer) {
        puts( "Memory could not be allocated.\nExiting..." );
        exit( EXIT_FAILURE );
    }
    buffer = malloc( (unsigned short)MAX_BUFFER_SIZE );
    if (NULL == buffer) {
        puts( "Memory could not be allocated.\nExiting..." );
        exit( EXIT_FAILURE );
    }
    memset( buffer, (int)0, data_size );

    /*
     * Initialize the conversation. This gives us a conversation ID
     * that we use for all subsequent CPI-C calls.
     */
    cminit( conv_id, (unsigned char *)BLANK_SYM_DEST_NAME, &rc );
    if (CM_OK != rc) {
        printf( "Error in cminit. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the partner TP name.
     */
    strcpy( ptp_name, PARTNER_TP_NAME );
    length = strlen( ptp_name );
    cmstpn( conv_id, ptp_name, &length, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmstpn. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the partner LU name to what the user specified on the command line.
     */
    cmspln( conv_id, partner_lu_name, &plu_name_length, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmspln. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the mode name.
     */
    strcpy( mode_name, MODE_NAME );
    length = strlen( mode_name );
    cmsmn( conv_id, mode_name, &length, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsmn. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the conversation security type to none.
     */
    security_type = CM_SECURITY_NONE;
    cmscst( conv_id, &security_type, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmscst. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the conversation type.
     */
    conv_type = CM_MAPPED_CONVERSATION;
    cmsct( conv_id, &conv_type, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsct. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the send/receive mode.
     */
    send_receive_mode = XC_FDX_OR_FDX_SIM;
    cmssrm( conv_id, &send_receive_mode, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmssrm. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Allocate the conversation.
     */
    cmallc( conv_id, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmallc. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set the send and receive queues to be non-blocking.  In order for
     * the calls to return their results, the CMWCMP call must be made.
     */
    conv_queue = CM_SEND_QUEUE;
    proc_mode = CM_NON_BLOCKING;
    user_field.value = CM_SEND_QUEUE;
    cmsqpm( conv_id,
            &conv_queue,
            &proc_mode,
            (char *)&user_field,
            &send_queue_op_id,
            &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsqpm. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }
    
    conv_queue = CM_RECEIVE_QUEUE;
    proc_mode = CM_NON_BLOCKING;
    user_field.value = CM_RECEIVE_QUEUE;
    cmsqpm( conv_id,
            &conv_queue,
            &proc_mode,
            (char *)&user_field,
            &receive_queue_op_id,
            &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsqpm. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }


    /*
     * Set the send type to indicate we want to flush. With full-duplex,
     * the receive that follows our send will not flush our send data.
     */
    send_type = CM_SEND_AND_FLUSH;
    cmsst( conv_id, &send_type, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsst. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Send data_size bytes of data, the first part of which is the
     * string to output on the other side.  This send call may or
     * may not complete...
     */
    strcpy( buffer, BURP_MESSAGE );
    length = data_size;
    cmsend( conv_id,
            buffer,
            &length,
            &rts_received,
            &send_rc );
    if (CM_OPERATION_INCOMPLETE != send_rc &&
        CM_OK != send_rc) {
        printf( "Error in cmsend. CPI-C rc = %lu", send_rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Receive back the useless data from our partner.  This *will*
     * be incomplete...
     */
    cmrcv( conv_id,                  /* Receive Data                  */
           receive_buffer,           /* Data Pointer                  */
           &max_rcv_len,             /* Size of Data Buffer           */
           &data_received,           /* returned - data received      */
           &received_len,            /* returned - length of data     */
           &status_received,         /* returned - status received    */
           &rts_received,            /* returned - request to send    */
           &receive_rc );
    if (CM_OPERATION_INCOMPLETE != receive_rc) {
        printf( "Error in cmrcv. CPI-C rc = %lu", receive_rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Wait for the calls to complete.
     * The send call should complete first.
     */
    outstanding_op_id_count = 0;
    if (CM_OPERATION_INCOMPLETE == send_rc) {
        outstanding_op_id_list[ outstanding_op_id_count++ ] = send_queue_op_id;
    }
    if (CM_OPERATION_INCOMPLETE == receive_rc) {
        outstanding_op_id_list[ outstanding_op_id_count++ ] =
                                             receive_queue_op_id;
    }
    
    timeout = -1;                       /* An indefinite wait */

    /* Assert( outstanding_op_id_count > 0 ); */
    
    /* Note that with the OS/2 CPI-C, only one call completes at a time. */
    cmwcmp( outstanding_op_id_list,
            &outstanding_op_id_count,
            &timeout,
            output_op_id_list,
            &output_op_id_count,
            (CM_USER_FIELD *)output_user_field_list,
            &rc );
    if (CM_OK != rc) {
        printf( "Error in cmwcmp. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Determine which call finished from the user_field that was filled
     * in by the CMWCMP call.
     */
    if (CM_SEND_QUEUE == output_user_field_list[0].value) {
        if (CM_OK != send_rc) {
            printf( "Error in cmsend. CPI-C rc = %lu", send_rc );
            exit( EXIT_FAILURE );
        }
        /* Now, wait for the receive call to complete */
        outstanding_op_id_count = 1;
        timeout = -1;
        outstanding_op_id_list[0] = receive_queue_op_id;
        
        cmwcmp( outstanding_op_id_list,
                &outstanding_op_id_count,
                &timeout,
                output_op_id_list,
                &output_op_id_count,
                (CM_USER_FIELD *)output_user_field_list,
                &rc );
        if (CM_OK != rc) {
            printf( "Error in cmwcmp. CPI-C rc = %lu", rc );
            exit( EXIT_FAILURE );
        }
    }
    
    /* The receive queue should have finished */
    if (CM_RECEIVE_QUEUE != output_user_field_list[0].value) {
        printf( "Error in cmwcmp.  Receive queue did not complete." );
        exit( EXIT_FAILURE );
    }

    if (CM_OK != receive_rc) {
        printf( "Error in cmrcv. CPI-C rc = %lu", receive_rc );
        exit( EXIT_FAILURE );
    }
        
    /*
     * Check to make sure the data matches what we sent
     */
    if ((received_len != length) ||
        (memcmp( receive_buffer, buffer, (size_t)received_len ))) {
        /* Ooops!  Something went wrong with the data */
        printf( "Error in data verification" );
        exit( EXIT_FAILURE );
    }

    /*
     * Set our send queue back to blocking.
     */
    conv_queue = CM_SEND_QUEUE;
    proc_mode = CM_BLOCKING;
    user_field.value = CM_SEND_QUEUE;
    cmsqpm( conv_id,
            &conv_queue,
            &proc_mode,
            (char *)&user_field,
            &send_queue_op_id,
            &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsqpm. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }
    
    /*
     * Set the deallocate type so our deallocate will flush.
     */
    deallocate_type = CM_DEALLOCATE_FLUSH;
    cmsdt( conv_id, &deallocate_type, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsdt. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Deallocate our side of the conversation.
     */
    cmdeal( conv_id, &rc );
    if (CM_OK != rc) {
        printf( "Error in cmdeal. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    /*
     * Set our receive queue back to blocking.
     */
    conv_queue = CM_RECEIVE_QUEUE;
    proc_mode = CM_BLOCKING;
    user_field.value = CM_RECEIVE_QUEUE;
    cmsqpm( conv_id,
            &conv_queue,
            &proc_mode,
            (char *)&user_field,
            &send_queue_op_id,
            &rc );
    if (CM_OK != rc) {
        printf( "Error in cmsqpm. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }
    
    /*
     * Receive an indication that our partner has deallocated his side
     * of the conversation.
     */
    cmrcv( conv_id,                  /* Receive Data                  */
           buffer,                   /* Data Pointer                  */
           &max_rcv_len,             /* Size of Data Buffer           */
           &data_received,           /* returned - data received      */
           &received_len,            /* returned - length of data     */
           &status_received,         /* returned - status received    */
           &rts_received,            /* returned - request to send    */
           &rc );

    if (CM_DEALLOCATED_NORMAL != rc) {
        printf( "Error in cmrcv. CPI-C rc = %lu", rc );
        exit( EXIT_FAILURE );
    }

    puts( "Hope that helps.\n" );
    
    exit( EXIT_SUCCESS );
}


void process_arguments( int argc, char *argv[] )
{
    int opt;
    int dest_set = 0;

    partner_lu_name[0] = '\0';
    
    /*
     * GETOPT is an easy way to parse command line arguments
     * Each parameter which can have a flag is passed in the third argument
     * to getopt.  Getopt returns the character of the flag on the command
     * line and sets optarg to point to the value associated with the flag.
     * optind is the index of the argument that getopt is currently processing.
     */

    while (optind != argc) {
        opt = getopt( argc, argv, "?s:n:" );
        switch (opt) {
        case EOF:
            optarg = argv[optind];
            if ('?' == optarg[0]) {
                puts( cmd_line );
                puts( explain );
                exit( EXIT_FAILURE );
            }
            optind++;
            if (dest_set) {
                puts( "\nOnly one partner LU name may be specified." );
                exit( EXIT_FAILURE );
            }
            else {
                dest_set = 1;
                plu_name_length = strlen( optarg );

                if (plu_name_length > MAX_TP_NAME_LEN) {
                    printf(
                        "\nThe destination you specified is too long:\n%s\n",
                         optarg);
                    printf(
                        "The maximum length allowed is %hu characters.\n",
                         ((unsigned short)MAX_TP_NAME_LEN-1) );
                    exit( EXIT_FAILURE );
                }
                strcpy( partner_lu_name, optarg );
            }
            break;

        case 's':
            /* Change the size of the data to send on the ping */
            data_size = atoi( optarg );
            break;

        case 'n':
            /* Change the number of times to restart the server */
            number_of_iterations = atoi( optarg );
            break;

        case '?':
            puts( cmd_line );
            puts( explain );
            exit( EXIT_FAILURE );
            break;
        default:
            puts( "Invalid flag.  Use BURP -? for usage\n" );
            exit( EXIT_FAILURE );
        }
    }
    
    if (!strlen(partner_lu_name)) {
        puts( "\nYou must specify a partner LU name." );
        puts( cmd_line );
        puts( explain );
        exit( EXIT_FAILURE );
    }
}
